'''
/***************************************************************************
Name		     : FireMappingToolWindow
Description          : Fire Mapping Tool Window functions for QGIS FMT3 Plugin
copyright            : (C) 2018 by Cheryl Holen
Created              : Sep 06, 2018 - Adapted from QGIS 2.x version
Updated              : Oct 11, 2018 - Fixed single scene check, perimeter scene
                       problem, added popups upon completion of steps,
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
'''
from PyQt5 import QtCore
from PyQt5 import QtWidgets
from PyQt5.QtWidgets import QMessageBox, QTableWidgetItem, QHeaderView, QDialog
from PyQt5.QtWidgets import QMainWindow, QAbstractItemView, QFileDialog
from qgis.gui import QgsStatusBar

from qgis.core import QgsProject
from qgis.core import QgsMessageLog, Qgis
from qgis.utils import iface

import configparser
import datetime
import getpass
import glob
import os
import re
import shutil
import webbrowser
from urllib.parse import urlparse, urlunparse

from .Ui_FireMappingTool import Ui_FireMappingTool
from .Ui_AddFire import Ui_AddFire
from .Ui_AddVectorStyle import Ui_AddVectorStyle
from .Ui_AddWMSLayer import Ui_AddWMSLayer
from .Ui_SearchByName import Ui_SearchByName
from .Ui_SearchByDate import Ui_SearchByDate
from .Ui_SearchByPathRowYear import Ui_SearchByPathRowYear
from .Ui_SearchByYear import Ui_SearchByYear
from .Ui_CreateMapping import Ui_CreateMapping
from .Ui_Preferences import Ui_Preferences
from .Ui_PreprocessImagery import Ui_PreprocessImagery
from .ScenePrep import ScenePrep
from .FirePrep import FirePrep
from .SubsetProcess import Subset
from .RdNBR import RdNBR
from .ThresholdProcess import ThresholdProcess
from .GenerateMetadata import GenerateMetadata
from .PreprocessData import PreprocessData
from .Utilities import Utilities
from FMT3 import constants as const

# global vars
date_format = '%Y-%m-%d'

db_dir = os.path.dirname(os.path.realpath(__file__))
db_fname = 'FireInfo.sqlite'
db_file = os.path.join(db_dir, db_fname)

status = ['', 'in-progress', 'complete']


# Create the dialog for Fire Mapping Tool
class FireMappingToolWindow(QMainWindow):
    config_filename = 'Config.ini'
    analysis_type = ['', 'dnbr', 'dndvi', 'nbr', 'ndvi']
    perim_conf = ['', 'High', 'Low']
    fireLabels = ['Name', 'Mapping Status', 'Acres', 'Event Date', 'EventID',
                  'P/R', 'Latitude', 'Longitude', 'Report Date', 'id']
    mapLabels = ['Id', 'Assessment Type', 'Pre-fire Scene',
                 'Post-fire Scene', 'Perimeter Scene', 'Mapper',
                 'Date Created']
    mapTableColCt = 7
    fire_table_col_ct = 10
    evt_prods_dir = 'event_prods'
    fire_dir = 'fire'
    mtbs_pref = 'mtbs_'
    pr_div = '/'
    nd_thresh = '-970'

    def __init__(self):
        QMainWindow.__init__(self)
        self.ui = Ui_FireMappingTool()
        self.ui.setupUi(self)
        self.setWindowTitle(const.GUI_TITLE)
        ConfigDir = os.path.dirname(os.path.realpath(__file__))
        self.ConfigFile = os.path.join(ConfigDir, self.config_filename)
        config = configparser.ConfigParser()
        config.read(self.ConfigFile)
        self.img_src = config.get('config', 'img_src').strip()
        if not self.img_src.endswith(os.sep):
            self.img_src += os.sep
        self.scene_src = config.get('config', 'base_dir').strip()
        if not self.scene_src.endswith(os.sep):
            self.scene_src += os.sep
        self.ndvi_url = config.get('config', 'ndvi_url').strip()
        self.vectors_styles_files_list = config.get(
            'config', 'vectors_styles').strip().splitlines()
        self.wms_layer_files_list = config.get(
            'config', 'wms_layers').strip().splitlines()
        self.ui.createMapButton.clicked.connect(self.create_mapping)
        self.ui.ndviButton.clicked.connect(self.create_ndvi)
        self.ui.actionAdd_Fire.triggered.connect(self.add_fire)
        self.ui.actionSearch_By_Name.triggered.connect(self.search_by_name)
        self.ui.actionSearch_By_Date.triggered.connect(self.search_by_date)
        self.ui.actionSearch_By_Path_Row_and_Year.triggered.connect(
                self.search_by_path_row_year)
        self.ui.actionSearch_By_Year.triggered.connect(self.search_by_year)
        self.ui.clearButton.clicked.connect(self.clear_fire)
        self.ui.maptableWidget.setSelectionBehavior(
                QAbstractItemView.SelectRows)
        self.ui.maptableWidget.setSelectionMode(
                QAbstractItemView.SingleSelection)
        self.ui.maptableWidget.itemClicked.connect(self.populate_data)
        self.ui.sceneprepButton.clicked.connect(self.scene_prep)
        self.ui.fireprepButton.clicked.connect(self.fire_prep)
        self.ui.subsetButton.clicked.connect(self.clip_subset)
        self.ui.delineateButton.clicked.connect(self.load_qgs_project)
        self.ui.openeventButton.clicked.connect(self.open_event_folder)
        self.ui.shapeButton.clicked.connect(self.write_perimeter_comment)
        self.ui.dNBRButton.clicked.connect(self.create_rdnbr)
        self.ui.threshButton.clicked.connect(self.threshold_image)
        self.ui.metaButton.clicked.connect(self.generate_metadata)
        self.ui.qacheckButton.clicked.connect(self.qa_checklist)
        self.ui.deleteButton.clicked.connect(self.delete_mapping)
        self.ui.updatemappingButton.clicked.connect(self.update_mapping)
        self.ui.generateButton.hide()
        self.ui.qacheckButton.hide()
        self.ui.preprocessDataButton.clicked.connect(self.preprocess_data)
        self.ui.actionChange_Preferences.triggered.connect(self.change_preferences)

        for i in self.analysis_type:
            self.ui.analysisBox.addItem(i)

        for j in self.perim_conf:
            self.ui.perimeterBox.addItem(j)

        for k in status:
            self.ui.mappingBox.addItem(k)
        self.ndvi_year = None
        self.ndvi_pathrow = None
        self.event_id = None
        self.ndvi_event_date = None
        self.utils = Utilities()
        self.status_bar = QgsStatusBar()

    def create_mapping(self):
        ''' Create Pre/Post fire DB entries depending on assessment type. '''
        if not self.ui.firetableWidget.currentItem():
            QMessageBox.critical(self,
                                 'Error!',
                                 'You must select a fire.',
                                 QMessageBox.Ok)
            return
        create_mapping_dlg = CreateMappingDialog(self.img_src)
        create_mapping_dlg.show()
        if create_mapping_dlg.exec_():
            sql = r"SELECT MAX(id) AS id FROM 'Mappings'"
            fetch = self.utils.run_query(sql)
            if fetch[0] is None:
                mapId = 1
            else:
                mapId = fetch[0] + 1
            mapper = getpass.getuser()
            create_date = datetime.datetime.now()
            row = self.ui.firetableWidget.currentItem().row()
            column_ct = self.ui.firetableWidget.columnCount()
            for i in range(column_ct):
                column_name =\
                    self.ui.firetableWidget.horizontalHeaderItem(i).text()
                if column_name == self.fireLabels[9]:
                    col = i
                    break
            fire_id = str(self.ui.firetableWidget.item(row, col).text())
            current_row_ct = self.ui.maptableWidget.rowCount()
            self.ui.maptableWidget.insertRow(current_row_ct)

            self.ui.maptableWidget.setItem(
                current_row_ct, 0, QTableWidgetItem(str(mapId)))
            self.ui.maptableWidget.setItem(
                current_row_ct, 1,
                QTableWidgetItem(str(create_mapping_dlg.assess_strategy)))
            if create_mapping_dlg.pre_supp_img is None:
                self.ui.maptableWidget.setItem(
                  current_row_ct, 2, QTableWidgetItem(' '))
            else:
                self.ui.maptableWidget.setItem(
                  current_row_ct, 2,
                  QTableWidgetItem(str(create_mapping_dlg.pre_supp_img)))
            if create_mapping_dlg.post_img is None:
                self.ui.maptableWidget.setItem(
                  current_row_ct, 3, QTableWidgetItem(' '))
            else:
                self.ui.maptableWidget.setItem(
                  current_row_ct, 3,
                  QTableWidgetItem(str(create_mapping_dlg.post_img)))
            if create_mapping_dlg.peri_img is None:
                self.ui.maptableWidget.setItem(
                  current_row_ct, 4, QTableWidgetItem(' '))
            else:
                self.ui.maptableWidget.setItem(
                  current_row_ct, 4,
                  QTableWidgetItem(str(create_mapping_dlg.peri_img)))
            self.ui.maptableWidget.setItem(
                current_row_ct, 5, QTableWidgetItem(str(mapper)))
            self.ui.maptableWidget.setItem(
                current_row_ct, 6, QTableWidgetItem(str(create_date)))
            mapping_status = ''
            params = (mapId, fire_id, create_mapping_dlg.assess_strategy,
                      create_mapping_dlg.pre_supp_img,
                      create_mapping_dlg.post_img,
                      create_mapping_dlg.peri_img,
                      create_mapping_dlg.pre_supp_sensor,
                      create_mapping_dlg.post_sensor,
                      create_mapping_dlg.peri_sensor,
                      create_mapping_dlg.prefire_date,
                      create_mapping_dlg.postfire_date,
                      create_mapping_dlg.peri_date, mapper,
                      create_date,
                      create_mapping_dlg.single_scene,
                      mapping_status,
                      create_mapping_dlg.pre_path,
                      create_mapping_dlg.pre_row,
                      create_mapping_dlg.post_path,
                      create_mapping_dlg.post_row,
                      create_mapping_dlg.assess_strategy)
            if (create_mapping_dlg.assess_strategy == 'Initial' or
                    create_mapping_dlg.assess_strategy == 'Extended'):
                sql = (r"""INSERT INTO 'Mappings'
                         (id, fire_id, strategy, prefire_scene_id,
                         postfire_scene_id, perimeter_scene_id,
                         prefire_sensor, postfire_sensor,
                         perimeter_sensor, prefire_date,
                         postfire_date, perimeter_date,
                         username, created_at, single_scene, status,
                         prefire_path, prefire_row, postfire_path,
                         postfire_row, predicted_strategy) VALUES(
                         ?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)""")
            else:
                sql = (r"""INSERT INTO 'Mappings'
                         (id, fire_id, strategy, supplementary_scene_id,
                         postfire_scene_id, perimeter_scene_id,
                         supplementary_sensor, postfire_sensor,
                         perimeter_sensor, supplementary_date,
                         postfire_date, perimeter_date, username,
                         created_at, single_scene, status, prefire_path,
                         prefire_row, postfire_path, postfire_row,
                         predicted_strategy) VALUES(
                         ?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)""")
            self.utils.write_db(sql, params)

    def add_fire(self):
        ''' Open dialog to add a fire '''
        add_fire_dlg = AddFireDialog()
        add_fire_dlg.show()
        add_fire_dlg.exec_()

    def create_ndvi(self):
        ''' Open url to display NDVI Graph for corresponding path/row '''
        if (self.ndvi_year is None or
                self.ndvi_pathrow is None or self.ndvi_event_date is None):
            QMessageBox.critical(self,
                                 'Error!',
                                 'You must select a fire.',
                                 QMessageBox.Ok)
            return
        var = 1
        scheme, netloc, path, params, query, fragment = urlparse(self.ndvi_url)
        # can only have 2 digit path rows for ndvi
        temp_path = self.ndvi_pathrow[1:3] + self.ndvi_pathrow[5:7]
        join_string = ('year=' + self.ndvi_year +
                       '&pathrow=' + temp_path +
                       '&eventDate=' + self.ndvi_event_date)
        new_query = query.replace(query, join_string)
        url = urlunparse((scheme, netloc, path, params, new_query, fragment))
        webbrowser.open(url, new=var)

    def clear_fire(self):
        ''' Clear the fire table '''
        self.ui.firetableWidget.setRowCount(0)
        self.ui.incidentCount.setText('')

    def search_fires(self, search_type, param):
        ''' Database query function for fires  '''
        if search_type == 'name':
            sql = (r"""SELECT incident_name, mapping_status,
                   area_burned, ig_date, event_id, path1, row1,
                   ig_lat, ig_long, report_date, id FROM 'Fires'
                    WHERE incident_name LIKE '%""" + param[0] + "%'")
        elif search_type == 'date':
            sql = (r"""SELECT incident_name, mapping_status,
                   area_burned, ig_date, event_id, path1, row1,
                   ig_lat, ig_long, report_date, id,
                   expected_containment_date,
                   actual_containment_date FROM 'Fires' WHERE """ +
                   param[0] + " BETWEEN '" +
                   param[1] + "' AND '" +
                   param[2] + "'")
        elif search_type == 'pathrowyear':
            sql = (r"""SELECT incident_name, mapping_status,
                   area_burned, ig_date, event_id, path1, row1,
                   ig_lat, ig_long, report_date, id FROM 'Fires'
                    WHERE strftime('%Y',ig_date) = '""" +
                   param[0] + "' AND path1 = '" +
                   param[1] + "' AND row1 = '" +
                   param[2] + "'")
        elif search_type == 'year':
            sql = (r"""SELECT incident_name, mapping_status,
                   area_burned, ig_date, event_id, path1, row1,
                   ig_lat, ig_long, report_date, id FROM 'Fires'
                    WHERE strftime('%Y',ig_date) = '""" + param[0] + "'")

        fetch = self.utils.run_query(sql, 1)
        # TODO:  This is not working, how to get table to fill widget?
        self.ui.firetableWidget.setSizeAdjustPolicy(
                QtWidgets.QAbstractScrollArea.AdjustToContents)
        self.ui.firetableWidget.setRowCount(0)
        self.ui.firetableWidget.setColumnCount(self.fire_table_col_ct)
        self.ui.firetableWidget.setEditTriggers(
                QAbstractItemView.NoEditTriggers)
        self.ui.firetableWidget.setHorizontalHeaderLabels(self.fireLabels)
        self.ui.firetableWidget.setSortingEnabled(True)
        self.ui.firetableWidget.resizeColumnsToContents()
        self.ui.firetableWidget.resizeRowsToContents()
        self.ui.firetableWidget.setSelectionBehavior(
                QAbstractItemView.SelectRows)
        self.ui.firetableWidget.setSelectionMode(
                QAbstractItemView.SingleSelection)
        self.ui.firetableWidget.itemClicked.connect(self.select_mapping)
        self.ui.maptableWidget.setColumnCount(self.mapTableColCt)
        self.ui.maptableWidget.horizontalHeader().setSectionResizeMode(
                QHeaderView.Stretch)
        self.ui.maptableWidget.setHorizontalHeaderLabels(self.mapLabels)
        temp_id = ''
        for fire_entry in fetch:
            current_row_ct = self.ui.firetableWidget.rowCount()
            self.ui.firetableWidget.insertRow(current_row_ct)
            self.ui.firetableWidget.setItem(
                    current_row_ct, 0, QTableWidgetItem(str(fire_entry[0])))
            self.ui.firetableWidget.setItem(
                    current_row_ct, 1, QTableWidgetItem(str(fire_entry[1])))
            self.ui.firetableWidget.setItem(
                    current_row_ct, 2, QTableWidgetItem(str(fire_entry[2])))
            self.ui.firetableWidget.setItem(
                    current_row_ct, 3, QTableWidgetItem(str(fire_entry[3])))
            self.ui.firetableWidget.setItem(
                    current_row_ct, 4, QTableWidgetItem(str(fire_entry[4])))
            self.ui.firetableWidget.setItem(
                    current_row_ct, 5, QTableWidgetItem(
                            str(fire_entry[5]) + self.pr_div +
                            str(fire_entry[6])))
            self.ui.firetableWidget.setItem(
                    current_row_ct, 6, QTableWidgetItem(str(fire_entry[7])))
            self.ui.firetableWidget.setItem(
                    current_row_ct, 7, QTableWidgetItem(str(fire_entry[8])))
            self.ui.firetableWidget.setItem(
                    current_row_ct, 8, QTableWidgetItem(str(fire_entry[9])))
            self.ui.firetableWidget.setItem(
                    current_row_ct, 9, QTableWidgetItem(str(fire_entry[10])))
            temp_id = fire_entry[10]

        total_cells = self.ui.firetableWidget.rowCount()
        self.ui.incidentCount.setText(str(total_cells))
        sql = (r"""SELECT id, strategy, prefire_scene_id,
               postfire_scene_id, perimeter_scene_id, username,
               created_at FROM 'Mappings' WHERE fire_id = '""" +
               str(temp_id) + "'")
        fetch = self.utils.run_query(sql, 1)
        self.ui.maptableWidget.setRowCount(0)
        self.ui.maptableWidget.setEditTriggers(
            QAbstractItemView.NoEditTriggers)
        for map_entry in fetch:
            current_row_ct = self.ui.maptableWidget.rowCount()
            self.ui.maptableWidget.insertRow(current_row_ct)
            for k in range(self.mapTableColCt):
                self.ui.maptableWidget.setItem(
                        current_row_ct, k,
                        QTableWidgetItem(str(map_entry[k])))

    def search_by_name(self):
        ''' Database query by name function for fires  '''
        name_search_dlg = SearchByNameDialog()
        name_search_dlg.show()
        if name_search_dlg.exec_():
            param_list = [str(name_search_dlg.selectedName)]
            self.search_fires('name', param_list)

    def search_by_date(self):
        ''' Database query by date function for fires  '''
        date_search_dlg = SearchByDateDialog()
        date_search_dlg.show()
        if date_search_dlg.exec_():
            param_list = [str(date_search_dlg.selectedCriteria),
                          str(date_search_dlg.selectedStartDate),
                          str(date_search_dlg.selectedEndDate)]
            self.search_fires('date', param_list)

    def search_by_path_row_year(self):
        ''' Database query by pathrowyear function for fires  '''
        pry_search_dlg = SearchByPathRowYearDialog()
        pry_search_dlg.show()
        if pry_search_dlg.exec_():
            param_list = [str(pry_search_dlg.selectedYear),
                          str(pry_search_dlg.selectedPath),
                          str(pry_search_dlg.selectedRow)]
            self.search_fires('pathrowyear', param_list)

    def search_by_year(self):
        ''' Database query by year function for fires  '''
        year_search_dlg = SearchByYearDialog()
        year_search_dlg.show()
        if year_search_dlg.exec_():
            param_list = [str(year_search_dlg.selectedYear)]
            self.search_fires('year', param_list)

    def select_mapping(self):
        ''' Display mapping event corresponding to selected fire event '''
        if not self.ui.firetableWidget.currentItem():
            QMessageBox.critical(self,
                                 'Error!',
                                 'You must select a fire.',
                                 QMessageBox.Ok)
            return
        row = self.ui.firetableWidget.currentItem().row()
        column_ct = self.ui.firetableWidget.columnCount()
        for i in range(column_ct):
            column_name =\
                self.ui.firetableWidget.horizontalHeaderItem(i).text()
            if column_name == self.fireLabels[9]:
                col = i
            elif column_name == self.fireLabels[3]:
                year = i
            elif column_name == self.fireLabels[5]:
                pr = i
            elif column_name == self.fireLabels[4]:
                e_id = i
        item = self.ui.firetableWidget.item(row, col).text()

        self.ndvi_year = self.ui.firetableWidget.item(row, year).text()[0:4]
        ndvi_pr = self.ui.firetableWidget.item(row, pr).text()

        tempList = ndvi_pr.split(self.pr_div)
        self.ndvi_pathrow = tempList[0].zfill(3) + '_' + tempList[1].zfill(3)

        self.ndvi_event_date = self.ui.firetableWidget.item(row, year).text()
        self.event_id = self.ui.firetableWidget.item(row, e_id).text()

        sql = (r"""SELECT id, strategy, prefire_scene_id, postfire_scene_id,
               perimeter_scene_id, username,created_at
                FROM 'Mappings' WHERE fire_id = '""" + str(item) + "'")

        fetch = self.utils.run_query(sql, 1)
        self.ui.maptableWidget.setRowCount(0)
        for map_entry in fetch:
            current_row_ct = self.ui.maptableWidget.rowCount()
            self.ui.maptableWidget.insertRow(current_row_ct)
            for k in range(self.mapTableColCt):
                self.ui.maptableWidget.setItem(
                        current_row_ct, k,
                        QTableWidgetItem(str(map_entry[k])))

    def get_mapping_id(self):
        ''' Get the mapping id from selected mapping '''
        mapping_id = None
        if self.ui.maptableWidget.currentItem() is None:
            QMessageBox.critical(self,
                                 'Error!',
                                 'You must select a mapping.',
                                 QMessageBox.Ok)
        else:
            row = self.ui.maptableWidget.currentItem().row()
            column_ct = self.ui.maptableWidget.columnCount()
            for i in range(column_ct):
                column_name =\
                    self.ui.maptableWidget.horizontalHeaderItem(i).text()
                if column_name == self.mapLabels[0]:
                    col = i
            mapping_id = self.ui.maptableWidget.item(row, col).text()

        return mapping_id

    def populate_data(self):
        ''' Loads mapping data from db to gui '''
        mapping_id = self.get_mapping_id()
        if mapping_id is None:
            return
        single_scene_ck = self.get_single_scene(mapping_id)
        if single_scene_ck:
            self.ui.dNBRButton.setEnabled(False)
            self.ui.sceneprepButton.setEnabled(False)
        else:
            self.ui.dNBRButton.setEnabled(True)
            self.ui.sceneprepButton.setEnabled(True)
        sql = (r"""SELECT analysis,dnbr_offset,perimeter_confidence,
               threshold1,threshold2,threshold3,no_data_thresh,
               greenness_thresh,status,single_scene,
               standard_deviation,perimeter_comments,
               mapping_comments FROM 'Mappings' WHERE id = """ + mapping_id)

        fetch = self.utils.run_query(sql, 1)
        for mapping_entry in fetch:
            if mapping_entry[0] == self.analysis_type[1]:
                self.ui.analysisBox.setCurrentIndex(1)
            elif mapping_entry[0] == self.analysis_type[2]:
                self.ui.analysisBox.setCurrentIndex(2)
            elif mapping_entry[0] == self.analysis_type[3]:
                self.ui.analysisBox.setCurrentIndex(3)
            elif mapping_entry[0] == self.analysis_type[4]:
                self.ui.analysisBox.setCurrentIndex(4)

            self.ui.offsetLineEdit.setText(mapping_entry[1])

            if mapping_entry[2] == self.perim_conf[1]:
                self.ui.perimeterBox.setCurrentIndex(1)
            elif mapping_entry[2] == self.perim_conf[2]:
                self.ui.perimeterBox.setCurrentIndex(2)

            self.ui.thresh1LineEdit.setText(str(mapping_entry[3]))
            self.ui.thresh2LineEdit.setText(str(mapping_entry[4]))
            self.ui.thresh3LineEdit.setText(str(mapping_entry[5]))
            self.ui.nothreshLineEdit.setText(str(mapping_entry[6]))
            self.ui.greenthreshLineEdit.setText(str(mapping_entry[7]))
            self.ui.sdLineEdit.setText(str(mapping_entry[10]))

            if mapping_entry[8] == status[1]:
                self.ui.mappingBox.setCurrentIndex(1)
            elif mapping_entry[8] == status[2]:
                self.ui.mappingBox.setCurrentIndex(2)
            else:
                self.ui.mappingBox.setCurrentIndex(0)

            if mapping_entry[9] == '1':
                self.ui.sceneprepButton.setEnabled(False)
            else:
                self.ui.sceneprepButton.setEnabled(True)
                self.ui.groupBox_3.setEnabled(True)

            self.ui.perimeterTextEdit.setPlainText(str(mapping_entry[11]))
            self.ui.mappingTextEdit.setPlainText(str(mapping_entry[12]))

        QgsProject.instance().clear()

    def scene_prep(self):
        ''' Creates mask/dNBR image '''
        mapping_id = self.get_mapping_id()
        if not mapping_id:
            return

        project = QgsProject.instance()
        project_name = project.absoluteFilePath()
        if project_name != '':
            if project.isDirty():
                project.write(project_name)
        project.clear()

        single_scene_ck = self.get_single_scene(mapping_id)
        if single_scene_ck:
            QMessageBox.critical(None,
                                 'Error!',
                                 ('Mapping is single scene.\n'
                                  'Exiting scene prep.'),
                                 QMessageBox.Ok)
            return
        if self.ui.checkBox.isChecked():
            resp = QMessageBox.warning(None,
                                       'Warning',
                                       ('This process will overwrite '
                                        'the files.\n'
                                        'Do you want to proceed?\n'
                                        'Press OK to continue or '
                                        'Cancel to exit.'),
                                       QMessageBox.Ok,
                                       QMessageBox.Cancel)
            if resp == QMessageBox.Ok:
                overwrite = True
            else:
                QgsMessageLog.logMessage(r'Scene Prep cancelled',
                                         level=Qgis.Info)
                return
        else:
            overwrite = False
        self.disable_controls()
        scene_prep = ScenePrep(mapping_id, self.scene_src,
                               self.img_src, overwrite)
        scene_prep.scene_prep_process()
        self.enable_controls()
        QgsMessageLog.logMessage(r'Scene Prep completed',
                                 level=Qgis.Info)

    def fire_prep(self):
        ''' Creates the Qgis project files corresponding to the mapping id '''
        mapping_id = self.get_mapping_id()
        if mapping_id is None:
            return

        project = QgsProject.instance()
        project_name = project.absoluteFilePath()
        if project_name != '':
            if project.isDirty():
                project.write(project_name)
        project.clear()

        self.disable_controls()
        f_prep = FirePrep(mapping_id, self.scene_src, self.img_src,
                          self.vectors_styles_files_list,
                          self.wms_layer_files_list)
        f_prep.fire_prep_process()
        self.enable_controls()
        QgsMessageLog.logMessage(r'Fire Prep completed',
                                 level=Qgis.Info)

    def load_qgs_project(self):
        ''' loads the project files '''
        mapping_id = self.get_mapping_id()
        if mapping_id is None:
            return
        sql = "SELECT * FROM 'Mappings' WHERE id = " + mapping_id
        fetch = self.utils.run_query(sql)
        fire_id = str(fetch[1])
        sql = "SELECT * FROM 'Fires' WHERE id = " + fire_id
        fetch = self.utils.run_query(sql)
        mtbs_id = fetch[1]
        fire_date = fetch[4]
        fire_date_list = str(fire_date).split('-')
        fire_year = fire_date_list[0]

        # Make string to use with glob
        temp_string = os.path.join(self.scene_src,
                                   self.evt_prods_dir,
                                   self.fire_dir,
                                   fire_year,
                                   mtbs_id.lower(),
                                   (self.mtbs_pref + mapping_id),
                                   '*.qgs')
        for infile in glob.glob(temp_string):
            project_file = infile
        if project_file:
            project = QgsProject.instance()
            project.read(project_file)
            zoom_layer = project.mapLayersByName("Post Scene Refl")[0]
            canvas = iface.mapCanvas()
            # Zoom to the full extent of the layer
            canvas.setExtent(zoom_layer.extent())
            canvas.refresh()

            layers = project.layerTreeRoot().children()
            # Only show these layers as selected to start when loading project
            enabled_layers = ["Burned Area Bndy",
                              "Mask",
                              "Post Scene Refl",
                              "Post Scene NBR",
                              "Post Scene dNBR"]
            for layer in layers:
                if layer.name() not in enabled_layers:
                    layer.setItemVisibilityChecked(False)

        else:
            QMessageBox.warning(None,
                                r'No project file found',
                                r'Run "Fire Prep"',
                                QMessageBox.Ok)
            return

        QgsMessageLog.logMessage(r'Qgis Project files are loaded',
                                 level=Qgis.Info)
        QMessageBox.information(None,
                                r'Status',
                                r'Qgis Project files are loaded',
                                QMessageBox.Ok)

    def open_event_folder(self):
        ''' Open the mapping folder '''
        mapping_id = self.get_mapping_id()
        if mapping_id is None:
            return
        event_prods_path = os.path.join(self.scene_src,
                                        self.evt_prods_dir,
                                        self.fire_dir)
        sql = ("SELECT * FROM 'Mappings' WHERE id = " + mapping_id)
        fetch = self.utils.run_query(sql)
        fire_id = str(fetch[1])
        sql = ("SELECT * FROM 'Fires' WHERE id = " + str(fire_id))
        fetch = self.utils.run_query(sql)
        mtbs_id = fetch[1]
        fire_date = fetch[4]
        fire_date_list = str(fire_date).split('-')
        fire_year = fire_date_list[0]

        # Find and open mapping folder
        mapping_folder = os.path.join(event_prods_path,
                                      fire_year,
                                      mtbs_id.lower(),
                                      (self.mtbs_pref + mapping_id))
        if os.path.exists(mapping_folder):
            os.startfile(mapping_folder)
        else:
            QMessageBox.information(None,
                                    'Error',
                                    ('Cannot find mapping folder: ' +
                                     mapping_folder),
                                    QMessageBox.Ok)

    def clip_subset(self):
        ''' Calculate dNBR offset and threshold values '''
        mapping_id = self.get_mapping_id()
        if mapping_id is None:
            return

        project = QgsProject.instance()
        project_name = project.absoluteFilePath()
        if project_name != '':
            if project.isDirty():
                project.write(project_name)
        project.clear()

        self.disable_controls()
        sbst = Subset(mapping_id, self.scene_src, self.img_src)
        dnbroff, th1, th2, th3, gth, sdoff, bbx, bby = sbst.process_subset()

        self.ui.offsetLineEdit.setText(str(dnbroff))
        self.ui.thresh1LineEdit.setText(str(th1))
        self.ui.thresh2LineEdit.setText(str(th2))
        self.ui.thresh3LineEdit.setText(str(th3))
        self.ui.nothreshLineEdit.setText(self.nd_thresh)
        self.ui.greenthreshLineEdit.setText(str(gth))
        self.ui.sdLineEdit.setText(str(sdoff))

        sql = (r"UPDATE 'Mappings' SET burn_bndy_alb_x = '" +
               bbx + "',burn_bndy_alb_y = '" + bby +
               "' WHERE id = '" + str(mapping_id) + "'")
        self.utils.write_db(sql)
        self.enable_controls()
        QgsMessageLog.logMessage(r'Subset completed',
                                 level=Qgis.Info)

    def write_perimeter_comment(self):
        ''' Set the perimeter confidence columns '''
        mapping_id = self.get_mapping_id()
        if mapping_id is None:
            return

        sql = (r"UPDATE 'Mappings' SET perimeter_confidence = '" +
               self.ui.perimeterBox.currentText() +
               "',perimeter_comments = '" +
               self.ui.perimeterTextEdit.toPlainText() +
               "' WHERE id = '" + str(mapping_id) + "'")
        self.utils.write_db(sql)

        QMessageBox.information(None,
                                'Status',
                                'Perimeter comments updated',
                                QMessageBox.Ok)

    def create_rdnbr(self):
        ''' Create RdNBR image '''
        mapping_id = self.get_mapping_id()
        if mapping_id is None:
            return

        project = QgsProject.instance()
        project_name = project.absoluteFilePath()
        if project_name != '':
            if project.isDirty():
                project.write(project_name)
        project.clear()

        self.disable_controls()

        sql = (r"UPDATE 'Mappings' SET analysis = '" +
               str(self.ui.analysisBox.currentText()) +
               "',dnbr_offset = '" + str(self.ui.offsetLineEdit.text()) +
               "',standard_deviation = '" + str(self.ui.sdLineEdit.text()) +
               "' WHERE id = '" + str(mapping_id) + "'")
        self.utils.write_db(sql)
        rdnbr = RdNBR(mapping_id,
                      self.scene_src,
                      str(self.ui.offsetLineEdit.text()))
        rdnbr.rdnbr_process()
        self.enable_controls()
        QgsMessageLog.logMessage('RdNBR completed', level=Qgis.Info)
        QMessageBox.information(None,
                                'Status',
                                'RnDBR completed',
                                QMessageBox.Ok)

    def threshold_image(self):
        ''' Create final image after applying threshold values '''
        mapping_id = self.get_mapping_id()
        if mapping_id is None:
            return
        project = QgsProject.instance()
        project_name = project.absoluteFilePath()
        if project_name != '':
            if project.isDirty():
                project.write(project_name)
        project.clear()
        self.disable_controls()
        low = str(self.ui.thresh1LineEdit.text())
        mod = str(self.ui.thresh2LineEdit.text())
        high = str(self.ui.thresh3LineEdit.text())
        regrowth = str(self.ui.greenthreshLineEdit.text())

        sql = (r"UPDATE 'Mappings' SET threshold1 = '" + low +
               "',threshold2 = '" + mod +
               "',threshold3 = '" + high +
               "',no_data_thresh = '" + str(self.ui.nothreshLineEdit.text()) +
               "',greenness_thresh = '" + regrowth +
               "',mapping_comments = '" +
               str(self.ui.mappingTextEdit.toPlainText()) +
               "' WHERE id = '" + str(mapping_id) + "'")
        self.utils.write_db(sql)

        thresh = ThresholdProcess(mapping_id,
                                  self.scene_src,
                                  low,
                                  mod,
                                  high,
                                  regrowth)
        thresh.calculate_thresholds()
        self.enable_controls()
        QgsMessageLog.logMessage('Threshold completed', level=Qgis.Info)
        QMessageBox.information(None,
                                'Status',
                                'Threshold completed',
                                QMessageBox.Ok)

    def generate_metadata(self):
        ''' Generate Mapping Metadata '''
        mapping_id = self.get_mapping_id()
        if mapping_id is None:
            return
        self.disable_controls()
        md = GenerateMetadata(mapping_id, self.scene_src)
        md.generate_metadata_process()
        self.enable_controls()
        QgsMessageLog.logMessage('Metadata  completed', level=Qgis.Info)
        QMessageBox.information(None,
                                'Status',
                                'Metadata completed',
                                QMessageBox.Ok)

    def qa_checklist(self):
        ''' QA checklist '''
        QMessageBox.information(None,
                                'Information',
                                'QA has not been implemented yet.',
                                QMessageBox.Ok)

    def get_single_scene(self, mapping_id):
        '''
        Checks if assessment type is single scene

        :param string: Mapping id
        :type integer: 0 for not single scene, 1 for single scene
        '''
        single_scene_ck = None

        sql = (r"SELECT single_scene FROM 'Mappings' WHERE id = " + mapping_id)
        fetch = self.utils.run_query(sql)
        for i in fetch:
            single_scene_ck = str(i[0])
        if single_scene_ck == '1':
            ret_val = True
        else:
            ret_val = False
        return ret_val

    def delete_mapping(self):
        ''' Delete the mapping entry from DB '''
        mapping_id = self.get_mapping_id()
        if mapping_id is None:
            return

        sql = (r"DELETE FROM 'Mappings' WHERE id = '" + str(mapping_id) + "'")
        self.utils.write_db(sql)

        row = self.ui.maptableWidget.currentItem().row()
        self.ui.maptableWidget.removeRow(row)

    def update_mapping(self):
        ''' Update the mapping status '''
        mapping_id = self.get_mapping_id()
        if mapping_id is None:
            return
        if not self.ui.firetableWidget.currentItem():
            QMessageBox.critical(self,
                                 'Error!',
                                 'You must select a fire.',
                                 QMessageBox.Ok)
            return
        row = self.ui.firetableWidget.currentItem().row()
        column_ct = self.ui.firetableWidget.columnCount()
        for i in range(column_ct):
            column_name =\
                self.ui.firetableWidget.horizontalHeaderItem(i).text()
            if column_name == self.fireLabels[9]:
                col = i
        fire_id = self.ui.firetableWidget.item(row, col).text()

        sql = (r"UPDATE 'Fires' SET mapping_status = '" +
               str(self.ui.mappingBox.currentText()) +
               "' WHERE id = '" + str(fire_id) + "'")
        self.utils.write_db(sql)
        sql = (r"UPDATE 'Mappings' SET status = '" +
               str(self.ui.mappingBox.currentText()) +
               "' WHERE id = '" + str(mapping_id) + "'")
        self.utils.write_db(sql)
        self.dynamic_update()
        QMessageBox.information(None,
                                'Status',
                                'Mapping updated',
                                QMessageBox.Ok)

    def dynamic_update(self):
        ''' Update fire mapping status in GUI '''
        if not self.ui.firetableWidget.currentItem():
            QMessageBox.critical(self,
                                 'Error!',
                                 'You must select a fire.',
                                 QMessageBox.Ok)
            return
        sql = r"SELECT mapping_status FROM 'Fires'"
        fetch = self.utils.run_query(sql, 1)
        row = self.ui.firetableWidget.currentItem().row()

        for entry in fetch:
            self.ui.firetableWidget.setItem(
                    row, 1, QTableWidgetItem(str(entry[0])))
    
    def preprocess_data(self):
        '''Unzip or Untar data and prepare for processing'''
        preprocess_data = PreprocessDataDialog(self.scene_src)
        preprocess_data.show()
        preprocess_data.exec_()

    def enable_controls(self):
        ''' Enable controls '''
        self.ui.createMapButton.setEnabled(True)
        self.ui.sceneprepButton.setEnabled(True)
        self.ui.fireprepButton.setEnabled(True)
        self.ui.delineateButton.setEnabled(True)
        self.ui.subsetButton.setEnabled(True)
        self.ui.openeventButton.setEnabled(True)
        self.ui.checkBox.setEnabled(True)
        self.ui.deleteButton.setEnabled(True)
        self.ui.updatemappingButton.setEnabled(True)
        self.ui.qacheckButton.setEnabled(True)
        self.ui.clearButton.setEnabled(True)
        self.ui.ndviButton.setEnabled(True)
        self.ui.metaButton.setEnabled(True)
        self.ui.preprocessDataButton.setEnabled(True)
        self.ui.groupBox.setEnabled(True)
        self.ui.groupBox_3.setEnabled(True)
        self.ui.groupBox_4.setEnabled(True)
        self.ui.groupBox_5.setEnabled(True)
        self.ui.analysisBox.setEnabled(True)
        self.ui.perimeterBox.setEnabled(True)

    def disable_controls(self):
        ''' Disable controls '''
        self.ui.createMapButton.setEnabled(False)
        self.ui.sceneprepButton.setEnabled(False)
        self.ui.fireprepButton.setEnabled(False)
        self.ui.delineateButton.setEnabled(False)
        self.ui.subsetButton.setEnabled(False)
        self.ui.openeventButton.setEnabled(False)
        self.ui.checkBox.setEnabled(False)
        self.ui.deleteButton.setEnabled(False)
        self.ui.updatemappingButton.setEnabled(False)
        self.ui.qacheckButton.setEnabled(False)
        self.ui.clearButton.setEnabled(False)
        self.ui.ndviButton.setEnabled(False)
        self.ui.metaButton.setEnabled(False)
        self.ui.preprocessDataButton.setEnabled(False)
        self.ui.groupBox.setEnabled(False)
        self.ui.groupBox_3.setEnabled(False)
        self.ui.groupBox_4.setEnabled(False)
        self.ui.groupBox_5.setEnabled(False)
        self.ui.analysisBox.setEnabled(False)
        self.ui.perimeterBox.setEnabled(False)

    def change_preferences(self):
        '''
        Set preferances in the config.ini
        '''
        preferances_dlg = ChangePreferences(
            self.ConfigFile, self.img_src, self.scene_src)
        preferances_dlg.show()
        preferances_dlg.exec_()

        # After updating preferences we need to re-read the paths
        config = configparser.ConfigParser()
        config.read(self.ConfigFile)
        self.img_src = config.get('config', 'img_src').strip()
        if not self.img_src.endswith(os.sep):
            self.img_src += os.sep
        self.scene_src = config.get('config', 'base_dir').strip()
        if not self.scene_src.endswith(os.sep):
            self.scene_src += os.sep
        self.vectors_styles_files_list = config.get(
            'config', 'vectors_styles').strip().splitlines()
        self.wms_layer_files_list = config.get(
            'config', 'wms_layers').strip().splitlines()


class AddFireDialog(QDialog):
    states = {
            'Alabama': 'AL',
            'Alaska': 'AK',
            'Arizona': 'AZ',
            'Arkansas': 'AR',
            'California': 'CA',
            'Colorado': 'CO',
            'Connecticut': 'CT',
            'Delaware': 'DE',
            'District Of Columbia': 'DC',
            'Florida': 'FL',
            'Georgia': 'GA',
            'Hawaii': 'HI',
            'Idaho': 'ID',
            'Illinois': 'IL',
            'Indiana': 'IN',
            'Iowa': 'IA',
            'Kansas': 'KS',
            'Kentucky': 'KY',
            'Louisiana': 'LA',
            'Maine': 'ME',
            'Maryland': 'MD',
            'Massachusetts': 'MA',
            'Michigan': 'MI',
            'Minnesota': 'MN',
            'Mississippi': 'MS',
            'Missouri': 'MO',
            'Montana': 'MT',
            'Nebraska': 'NE',
            'Nevada': 'NV',
            'New Hampshire': 'NH',
            'New Jersey': 'NJ',
            'New Mexico': 'NM',
            'New York': 'NY',
            'North Carolina': 'NC',
            'North Dakota': 'ND',
            'Ohio': 'OH',
            'Oklahoma': 'OK',
            'Oregon': 'OR',
            'Pennsylvania': 'PA',
            'Puerto Rico': 'PR',
            'Rhode Island': 'RI',
            'South Carolina': 'SC',
            'South Dakota': 'SD',
            'Tennessee': 'TN',
            'Texas': 'TX',
            'Utah': 'UT',
            'Vermont': 'VT',
            'Virginia': 'VA',
            'Washington': 'WA',
            'West Virginia': 'WV',
            'Wisconsin': 'WI',
            'Wyoming': 'WY',
            'International': 'IT'}

    def __init__(self):
        QDialog.__init__(self)
        self.ui = Ui_AddFire()
        self.ui.setupUi(self)
        self.ui.igDateEdit.setCalendarPopup(True)

        for i in self.states.items():
            self.ui.stateBox.addItem(i[0])

        self.ui.acreSpinBox.setValue(999999)

        event_type = ['WF', 'RX', 'WFU', 'UNK', 'Other']
        for j in event_type:
            self.ui.eventBox.addItem(j)

        self.ui.eventIDButton.clicked.connect(self.save_fire_info)

    def save_fire_info(self):
        ''' Saves fire info to db '''
        utils = Utilities()
        sql = r"SELECT MAX(id) AS id FROM 'Fires'"
        fetch = utils.run_query(sql)
        if fetch[0] is None:
            fire_id = 1
        else:
            fire_id = fetch[0] + 1

        box_value = self.ui.stateBox.currentText()
        state = self.states.get(box_value)
        lat_decimal = self.ui.latSpinBox.value()
        long_decimal = self.ui.longSpinBox.value()
        iso_data_format = self.ui.igDateEdit.date().toString(QtCore.Qt.ISODate)
        s_date = iso_data_format.replace('-', '')
        # it is possible that we have both positive and negative lat annd long
        # but assume a single user will not have both, so we remove any '-'
        # for the event id
        s_lat = format(lat_decimal, '.3f')
        s_long = format(long_decimal, '.3f')

        eventID = (state +
                   s_lat.replace('.', '').replace('-', '').zfill(5) +
                   s_long.replace('.', '').replace('-', '').zfill(6) +
                   s_date)

        params = (str(fire_id),
                  eventID,
                  self.ui.nameEdit.text(),
                  self.ui.eventBox.currentText(),
                  iso_data_format,
                  s_lat,
                  s_long,
                  str(self.ui.acreSpinBox.value()),
                  self.ui.expectedDateEdit.text(),
                  self.ui.actualDateEdit.text(),
                  self.ui.stateBox.currentText(),
                  self.ui.adminEdit.text(),
                  self.ui.agencyEdit.text(),
                  self.ui.reportDateEdit.text(),
                  self.ui.commentEdit.text(),
                  self.ui.fuelsEdit.text(),
                  str(self.ui.pathSpinBox.value()),
                  str(self.ui.rowSpinBox.value()), status[1], 'NULL',)

        sql = (r"""INSERT INTO 'Fires' (id,event_id,incident_name,
               incident_type,ig_date,ig_lat,ig_long,area_burned,
               expected_containment_date,actual_containment_date,
               ig_state,ig_admin,ig_agency,report_date,comment,
               fuels,path1,row1,mapping_status,mapping_in_progress)
               VALUES(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)""")
        utils.write_db(sql, params)


class SearchByNameDialog(QDialog):
    def __init__(self):
        QDialog.__init__(self)
        self.ui = Ui_SearchByName()
        self.ui.setupUi(self)
        self.ui.searchButton.clicked.connect(self.search_by_event_name)

    def search_by_event_name(self):
        ''' Get the selected fire name to search on '''
        self.selectedName = self.ui.incidentLineEdit.text()
        QDialog.accept(self)


class SearchByDateDialog(QDialog):
    def __init__(self):
        QDialog.__init__(self)
        self.ui = Ui_SearchByDate()
        self.ui.setupUi(self)
        self.ui.startdateEdit.setCalendarPopup(True)
        self.ui.startdateEdit.setDate(QtCore.QDate.currentDate())
        self.ui.enddateEdit.setCalendarPopup(True)
        self.ui.enddateEdit.setDate(QtCore.QDate.currentDate())
        date_labels = ['ig_date',
                       'report_date',
                       'expected_containment_date',
                       'actual_containment_date']
        for param in date_labels:
            self.ui.dateFieldComboBox.addItem(param)
        self.ui.searchButton.clicked.connect(self.search_by_event_date)

    def search_by_event_date(self):
        ''' Get the selected fire dates to search on '''
        if self.ui.dateFieldComboBox.currentText() == 'ig_date':
            self.selectedCriteria = 'ig_date'
        elif self.ui.dateFieldComboBox.currentText() == 'report_date':
            self.selectedCriteria = 'report_date'
        elif (self.ui.dateFieldComboBox.currentText() ==
              'expected_containment_date'):
            self.selectedCriteria = 'expected_containment_date'
        elif (self.ui.dateFieldComboBox.currentText() ==
              'actual_containment_date'):
            self.selectedCriteria = 'actual_containment_date'
        self.selectedStartDate = self.ui.startdateEdit.date().toPyDate()
        self.selectedEndDate = self.ui.enddateEdit.date().toPyDate()
        QDialog.accept(self)


class SearchByPathRowYearDialog(QDialog):
    def __init__(self):
        QDialog.__init__(self)
        self.ui = Ui_SearchByPathRowYear()
        self.utils = Utilities()
        self.ui.setupUi(self)

        sql = r"SELECT DISTINCT strftime('%Y',ig_date) FROM 'Fires'"
        fetch = self.utils.run_query(sql, 1)
        year_list = []
        for years in fetch:
            year_list.append(years[0])
        year_list = sorted(set(year_list), key=None, reverse=True)
        for entries in year_list:
            self.ui.yearComboBox.addItem(str(entries))
        self.ui.searchButton.clicked.connect(
                self.search_by_event_path_row_year)

    def search_by_event_path_row_year(self):
        ''' Get the selected fire path row and year to search on '''
        self.selectedYear = self.ui.yearComboBox.currentText()
        self.selectedPath = self.ui.pathLineEdit.text()
        self.selectedRow = self.ui.rowLineEdit.text()
        QDialog.accept(self)


class SearchByYearDialog(QDialog):
    def __init__(self):
        QDialog.__init__(self)
        self.ui = Ui_SearchByYear()
        self.ui.setupUi(self)
        utils = Utilities()

        sql = (r"""SELECT DISTINCT strftime('%Y',ig_date) AS ig_date
                 FROM 'Fires'""")
        fetch = utils.run_query(sql, 1)
        year_list = []
        for years in fetch:
            year_list.append(years[0])
        year_list = sorted(set(year_list), key=None, reverse=True)
        for entries in year_list:
            self.ui.yearComboBox.addItem(str(entries))
        self.ui.searchButton.clicked.connect(self.search_by_event_year)

    def search_by_event_year(self):
        ''' Get the selected fire year to search on '''
        self.selectedYear = self.ui.yearComboBox.currentText()
        QDialog.accept(self)


class PreprocessDataDialog(QDialog):
    def __init__(self, scene_src):
        QDialog.__init__(self)
        self.ui = Ui_PreprocessImagery()
        self.ui.setupUi(self)

        self.scene_src = scene_src
        self.sensor = self.ui.sensorComboBox.currentText().lower()

        self.ui.sensorComboBox.currentIndexChanged.connect(self.updateComboBox)
        self.ui.selectButton.clicked.connect(self.select_all)
        self.ui.clearButton.clicked.connect(self.clear_selection)
        self.ui.processButton.clicked.connect(self.process)
        self.ui.reprojectCheckBox.setChecked(False)
        self.ui.stack10mSentinelCheckBox.setChecked(False)
        self.ui.stack10mSentinelCheckBox.setVisible(False)

        self.updateComboBox()

    def disable_controls(self):
        self.ui.selectButton.setEnabled(False)
        self.ui.clearButton.setEnabled(False)
        self.ui.cancelButton.setEnabled(False)
        self.ui.processButton.setEnabled(False)
        self.ui.reprojectCheckBox.setEnabled(False)
        self.ui.stack10mSentinelCheckBox.setEnabled(False)
        self.ui.sensorComboBox.setEnabled(False)
        self.ui.filesListWidget.setEnabled(False)
        self.ui.regionComboBox.setEnabled(False)
        self.ui.samplingComboBox.setEnabled(True)

    def enable_controls(self):
        self.ui.selectButton.setEnabled(True)
        self.ui.clearButton.setEnabled(True)
        self.ui.cancelButton.setEnabled(True)
        self.ui.processButton.setEnabled(True)
        self.ui.reprojectCheckBox.setEnabled(True)
        self.ui.stack10mSentinelCheckBox.setEnabled(True)
        self.ui.sensorComboBox.setEnabled(True)
        self.ui.filesListWidget.setEnabled(True)
        self.ui.regionComboBox.setEnabled(True)
        self.ui.samplingComboBox.setEnabled(True)

    def process(self):
        '''run preprocessing script on selected files'''
        file_list = self.ui.filesListWidget.selectedItems()
        input_list = []
        for file in file_list:
            input_list.append(os.path.join(self.scene_src, const.RAW_DIR,
                                           self.sensor, file.text()))
        output_dir = os.path.join(self.scene_src, const.SRC_DIR)

        region = self.ui.regionComboBox.currentText()
        if region == 'CONUS': area = 1
        elif region == 'Alaska': area = 2
        elif region == 'Hawaii': area = 3
        elif region == 'Puerto Rico': area = 4

        sampling = self.ui.samplingComboBox.currentText()
        if sampling == 'Nearest Neighbor': sample = 0
        elif sampling == 'Bilinear': sample = 1
        elif sampling == 'Cubic': sample = 2

        reproject = self.ui.reprojectCheckBox.isChecked()
        work_dir = os.path.join(self.scene_src, 'working')

        prc = PreprocessData(reproject, output_dir, self.scene_src,
                             area, sample, work_dir)
        complete = False

        if input_list:
            if self.ui.sensorComboBox.currentText() == const.LANDSAT:
                for entry in input_list:
                    complete = prc.prepare_landsat_file(entry)
            else:
                stack10mSentinel = self.ui.stack10mSentinelCheckBox.isChecked()
                for entry in input_list:
                    complete = prc.prepare_sentinel2_file(entry, stack10mSentinel)
            
        else:
            QMessageBox.information(None,
                                    u'No files selected to process',
                                    (u'File selection was empty. Please select file(s) to process'),
                                    QMessageBox.Ok)
        if complete:
            QMessageBox.information(None,
                                    u'Process files complete',
                                    u'Processing files completed',
                                    QMessageBox.Ok)
            QgsMessageLog.logMessage(r'Processing files completed',
                                    level=Qgis.Info)
            for filename in os.listdir(work_dir):
                file_path = os.path.join(work_dir, filename)
                try:
                    if os.path.isfile(file_path) or os.path.islink(file_path):
                        os.unlink(file_path)
                    elif os.path.isdir(file_path):
                        shutil.rmtree(file_path)
                except Exception as e:
                    QgsMessageLog.logMessage('Failed to delete %s. Reason: %s' % (file_path, e), level=Qgis.Error)
        return

    def updateComboBox(self):
        '''function to update window when sensor is changed'''
        self.sensor = self.ui.sensorComboBox.currentText().lower()
        self.ui.inputFolderLineEdit.setText(os.path.join(self.scene_src,
                                                         const.RAW_DIR,
                                                         self.sensor))
        self.input_path = self.ui.inputFolderLineEdit.text()
        if self.sensor == const.LANDSAT:
            self.ui.stack10mSentinelCheckBox.setVisible(False)
        else:
            self.ui.stack10mSentinelCheckBox.setVisible(True)

        self.listFiles()

    def listFiles(self):
        '''function to generate list of available .tar, .tar.gz, and .zip files'''
        file_lst = (glob.glob(os.path.join(self.input_path, '*.TAR')) +
                    glob.glob(os.path.join(self.input_path, '*.TAR.GZ')) +
                    glob.glob(os.path.join(self.input_path, '*.ZIP')))
        
        self.ui.filesListWidget.clear()

        for file in file_lst:
            file_name = os.path.basename(file)
            self.ui.filesListWidget.addItem(file_name)

    def select_all(self):
        '''function to select everything in file list'''
        self.ui.filesListWidget.selectAll()

    def clear_selection(self):
        '''function to clear selection from file list'''
        self.ui.filesListWidget.clearSelection()


class CreateMappingDialog(QDialog):
    def __init__(self, img_src):
        QDialog.__init__(self)
        self.ui = Ui_CreateMapping()
        self.ui.setupUi(self)
        self.utils = Utilities()
        self.img_src = img_src

        self.ui.assessmentComboBox.activated.connect(self.assessment_select)
        self.ui.preImagepushButton.clicked.connect(self.get_prefire_data)
        self.ui.postImagepushButton.clicked.connect(self.get_postfire_data)
        self.ui.periImagepushButton.clicked.connect(self.get_perifire_data)
        self.ui.saveButton.clicked.connect(self.save_mapping)
        self.assessment = [' ', 'Initial', 'Initial (SS)',
                      'Extended', 'Extended (SS)']
        for j in self.assessment:
            self.ui.assessmentComboBox.addItem(j)
        self.single_scene = None

    def assessment_select(self):
        ''' Select assessment type '''
        if (self.ui.assessmentComboBox.currentText() == 'Initial' or
                self.ui.assessmentComboBox.currentText() == 'Extended'):
            self.ui.labelImage.setText('Pre Image Folder')
            self.ui.labelSensor.setText('Pre Sensor Used')
            self.ui.labelDate.setText('Prefire Date')
            self.single_scene = 0
        elif (self.ui.assessmentComboBox.currentText() == 'Initial (SS)' or
              self.ui.assessmentComboBox.currentText() == 'Extended (SS)'):
            self.ui.labelImage.setText('Supplementary Image Folder')
            self.ui.labelSensor.setText('Supplementary Sensor Used')
            self.ui.labelDate.setText('Supplementary Date')
            self.single_scene = 1

    def get_prefire_data(self):
        ''' Browse and select '''
        folder_name = QFileDialog.getExistingDirectory(
            self,
            u'Select Prefire Image Folder',
            self.img_src)
        if folder_name != "":
            self.ui.preImagelineEdit.setText(
                self.utils.fix_os_sep_in_path(folder_name))
        self.prefire()

    def prefire(self):
        ''' Select prefire '''
        full_path = self.ui.preImagelineEdit.text().rstrip()
        # Remove '/' from end of string if it exists
        if full_path.endswith(os.sep):
            full_path = full_path[:-1]

        if not full_path.startswith(self.img_src):
            QMessageBox.critical(
                self, 'Error!',
                'File path is not in the img_src ' +
                'directory from the config.ini file',
                QMessageBox.Ok)
            self.ui.preImagelineEdit.setText("")
            self.ui.preDatelineEdit.setText("")
            self.ui.preSensorlineEdit.setText("")
        else:
            f_name = os.path.basename(full_path)
            try:
                # Landsat
                if f_name.startswith(const.SENSOR_ARRAY[0][0]):
                    self.ui.preSensorlineEdit.setText(const.SENSOR_ARRAY[0][1])
                    date_idx = 7
                elif f_name.startswith(const.SENSOR_ARRAY[1][0]):
                    self.ui.preSensorlineEdit.setText(const.SENSOR_ARRAY[1][1])
                    date_idx = 7
                elif f_name.startswith(const.SENSOR_ARRAY[2][0]):
                    self.ui.preSensorlineEdit.setText(const.SENSOR_ARRAY[2][1])
                    date_idx = 7
                elif f_name.startswith(const.SENSOR_ARRAY[3][0]):
                    self.ui.preSensorlineEdit.setText(const.SENSOR_ARRAY[3][1])
                    date_idx = 7
                # Sentinel2
                elif f_name.startswith(const.SENSOR_ARRAY[4][0]):
                    self.ui.preSensorlineEdit.setText(const.SENSOR_ARRAY[4][1])
                    date_idx = 6
                elif f_name.startswith(const.SENSOR_ARRAY[5][0]):
                    self.ui.preSensorlineEdit.setText(const.SENSOR_ARRAY[5][1])
                    date_idx = 6
                # Unknown
                else:
                    self.ui.preSensorlineEdit.setText("Unknown Sensor")
                    date_idx = None

                # Use the date_idx to get info from the folder name
                if date_idx:
                    year = int(f_name[date_idx:date_idx + 4])
                    month = int(f_name[date_idx + 4:date_idx + 6])
                    day = int(f_name[date_idx + 6:date_idx + 8])
                    date = datetime.datetime(year, month, day)
                    self.ui.preDatelineEdit.setText(date.strftime(date_format))
                else:
                    self.ui.preDatelineEdit.setText("Unknown Folder Name Format")

            except BaseException:
                self.ui.preDatelineEdit.setText("Unknown Folder Name Format")
                self.ui.preSensorlineEdit.setText("Unknown Folder Name Format")

    def get_postfire_data(self):
        ''' Browse and select '''
        folder_name = QFileDialog.getExistingDirectory(
            self,
            u'Select Postfire Image Folder',
            self.img_src)
        if folder_name != "":
            self.ui.postImagelineEdit.setText(
                self.utils.fix_os_sep_in_path(folder_name))
        self.postfire()

    def postfire(self):
        ''' Select postfire '''
        full_path = self.ui.postImagelineEdit.text().rstrip()
        # Remove '/' from end of string if it exists
        if full_path.endswith(os.sep):
            full_path = full_path[:-1]

        if not full_path.startswith(self.img_src):
            QMessageBox.critical(
                self, 'Error!',
                'File path is not in the img_src ' +
                'directory from the config.ini file',
                QMessageBox.Ok)
            self.ui.postImagelineEdit.setText("")
            self.ui.postDatelineEdit.setText("")
            self.ui.postSensorlineEdit.setText("")
        else:
            f_name = os.path.basename(full_path)
            try:
                # Landsat
                if f_name.startswith(const.SENSOR_ARRAY[0][0]):
                    self.ui.postSensorlineEdit.setText(const.SENSOR_ARRAY[0][1])
                    date_idx = 7
                elif f_name.startswith(const.SENSOR_ARRAY[1][0]):
                    self.ui.postSensorlineEdit.setText(const.SENSOR_ARRAY[1][1])
                    date_idx = 7
                elif f_name.startswith(const.SENSOR_ARRAY[2][0]):
                    self.ui.postSensorlineEdit.setText(const.SENSOR_ARRAY[2][1])
                    date_idx = 7
                elif f_name.startswith(const.SENSOR_ARRAY[3][0]):
                    self.ui.postSensorlineEdit.setText(const.SENSOR_ARRAY[3][1])
                    date_idx = 7
                # Sentinel2
                elif f_name.startswith(const.SENSOR_ARRAY[4][0]):
                    self.ui.postSensorlineEdit.setText(const.SENSOR_ARRAY[4][1])
                    date_idx = 6
                elif f_name.startswith(const.SENSOR_ARRAY[5][0]):
                    self.ui.postSensorlineEdit.setText(const.SENSOR_ARRAY[5][1])
                    date_idx = 6
                # Unknown
                else:
                    self.ui.postSensorlineEdit.setText("Unknown Sensor")
                    date_idx = None

                # Use the date_idx to get info from the folder name
                if date_idx:
                    year = int(f_name[date_idx:date_idx + 4])
                    month = int(f_name[date_idx + 4:date_idx + 6])
                    day = int(f_name[date_idx + 6:date_idx + 8])
                    date = datetime.datetime(year, month, day)
                    self.ui.postDatelineEdit.setText(date.strftime(date_format))
                else:
                    self.ui.postDatelineEdit.setText("Unknown Folder Name Format")

            except BaseException:
                self.ui.postDatelineEdit.setText("Unknown Folder Name Format")
                self.ui.postSensorlineEdit.setText("Unknown Folder Name Format")

    def get_perifire_data(self):
        ''' Browse and select '''
        folder_name = QFileDialog.getExistingDirectory(
            self,
            u'Select Perifire Image Folder',
            self.img_src)
        if folder_name != "":
            self.ui.periImagelineEdit.setText(
                self.utils.fix_os_sep_in_path(folder_name))
        self.perifire()

    def perifire(self):
        ''' Select perimeter fire '''
        full_path = self.ui.periImagelineEdit.text().rstrip()
        # Remove '/' from end of string if it exists
        if full_path.endswith(os.sep):
            full_path = full_path[:-1]

        if not full_path.startswith(self.img_src):
            QMessageBox.critical(
                self, 'Error!',
                'File path is not in the img_src ' +
                'directory from the config.ini file',
                QMessageBox.Ok)
            self.ui.periImagelineEdit.setText("")
            self.ui.periDatelineEdit.setText("")
            self.ui.periSensorlineEdit.setText("")
        else:
            f_name = os.path.basename(full_path)
            try:
                # Landsat
                if f_name.startswith(const.SENSOR_ARRAY[0][0]):
                    self.ui.periSensorlineEdit.setText(const.SENSOR_ARRAY[0][1])
                    date_idx = 7
                elif f_name.startswith(const.SENSOR_ARRAY[1][0]):
                    self.ui.periSensorlineEdit.setText(const.SENSOR_ARRAY[1][1])
                    date_idx = 7
                elif f_name.startswith(const.SENSOR_ARRAY[2][0]):
                    self.ui.periSensorlineEdit.setText(const.SENSOR_ARRAY[2][1])
                    date_idx = 7
                elif f_name.startswith(const.SENSOR_ARRAY[3][0]):
                    self.ui.periSensorlineEdit.setText(const.SENSOR_ARRAY[3][1])
                    date_idx = 7
                # Sentinel2
                elif f_name.startswith(const.SENSOR_ARRAY[4][0]):
                    self.ui.periSensorlineEdit.setText(const.SENSOR_ARRAY[4][1])
                    date_idx = 6
                elif f_name.startswith(const.SENSOR_ARRAY[5][0]):
                    self.ui.periSensorlineEdit.setText(const.SENSOR_ARRAY[5][1])
                    date_idx = 6
                # Unknown
                else:
                    self.ui.periSensorlineEdit.setText("Unknown Sensor")
                    date_idx = None

                # Use the date_idx to get info from the folder name
                if date_idx:
                    year = int(f_name[date_idx:date_idx + 4])
                    month = int(f_name[date_idx + 4:date_idx + 6])
                    day = int(f_name[date_idx + 6:date_idx + 8])
                    date = datetime.datetime(year, month, day)
                    self.ui.periDatelineEdit.setText(date.strftime(date_format))
                else:
                    self.ui.periDatelineEdit.setText("Unknown Folder Name Format")
                
            except BaseException:
                self.ui.periDatelineEdit.setText("Unknown Folder Name Format")
                self.ui.periSensorlineEdit.setText("Unknown Folder Name Format")

    def save_mapping(self):
        ''' Save mapping '''
        self.assess_strategy = self.ui.assessmentComboBox.currentText()
        if self.assess_strategy == self.assessment[0]:
            QMessageBox.critical(
                self, 'Error!',
                'Must select an Assessment Strategy',
                QMessageBox.Ok)
            return

        # Prefire
        if not self.ui.preImagelineEdit.text().rstrip() == '':
            pre_full_path = self.ui.preImagelineEdit.text().rstrip()
            # Remove '/' from end of string if it exists
            if pre_full_path.endswith(os.sep):
                pre_full_path = pre_full_path[:-1]
            self.pre_supp_img = os.path.basename(pre_full_path)
        else:
            self.pre_supp_img = None

        self.pre_path, self.pre_row = self.utils.get_path_row(self.pre_supp_img)

        # Postfire
        if not self.ui.postImagelineEdit.text().rstrip() == '':
            post_full_path = self.ui.postImagelineEdit.text().rstrip()
            # Remove '/' from end of string if it exists
            if post_full_path.endswith(os.sep):
                post_full_path = post_full_path[:-1]
            self.post_img = os.path.basename(post_full_path)
        else:
            self.post_img = None
            QMessageBox.critical(
                self, 'Error!',
                'No Post-fire Scene selected, ' +
                'Must select a post fire scene',
                QMessageBox.Ok)
            return

        self.post_path, self.post_row = self.utils.get_path_row(self.post_img)

        if self.assess_strategy == self.assessment[1] or\
                self.assess_strategy == self.assessment[3]:
            if self.pre_supp_img is None or self.post_img is None:
                QMessageBox.critical(
                self, 'Error!',
                'Assessment Type is not Single Scene (SS), ' +
                'must select a Post and Pre fire scene.',
                QMessageBox.Ok)
                return

        # Perimeter
        if not self.ui.periImagelineEdit.text().rstrip() == '':
            peri_full_path = self.ui.periImagelineEdit.text().rstrip()
            # Remove '/' from end of string if it exists
            if peri_full_path.endswith(os.sep):
                peri_full_path = peri_full_path[:-1]
            self.peri_img = os.path.basename(peri_full_path)
        else:
            self.peri_img = None

        self.peri_path, self.peri_row = self.utils.get_path_row(self.peri_img)

        # Sensor Info
        if self.pre_supp_img:
            self.pre_supp_sensor = self.ui.preSensorlineEdit.text()
            if self.pre_supp_sensor not in [i[1] for i in const.SENSOR_ARRAY]:
                QMessageBox.critical(
                    self, 'Error!',
                    'Unknown Pre Sensor\n' +
                    'Possible Sensors: ' + str([i[1] for i in const.SENSOR_ARRAY]),
                    QMessageBox.Ok)
                return
        else:
            self.pre_supp_sensor = None

        if self.post_img:
            self.post_sensor = self.ui.postSensorlineEdit.text()
            if self.post_sensor not in [i[1] for i in const.SENSOR_ARRAY]:
                QMessageBox.critical(
                    self, 'Error!',
                    'Unknown Post Sensor\n' +
                    'Possible Sensors: ' + str([i[1] for i in const.SENSOR_ARRAY]),
                    QMessageBox.Ok)
                return
        else:
            self.post_sensor = None
        if self.peri_img:
            self.peri_sensor = self.ui.periSensorlineEdit.text()
            if self.peri_sensor not in [i[1] for i in const.SENSOR_ARRAY]:
                QMessageBox.critical(
                    self, 'Error!',
                    'Unknown Perimeter Sensor\n' +
                    'Possible Sensors: ' + str([i[1] for i in const.SENSOR_ARRAY]),
                    QMessageBox.Ok)
                return
        else:
            self.peri_sensor = None
        # Check all sensors match
        sensors_match, sentinel = self.check_mapping_sensors_match(
            self.pre_supp_sensor, self.post_sensor, self.peri_sensor)
        if not sensors_match:
            QMessageBox.critical(
                self, 'Error!',
                'Sensors must all be the same, ' +
                'either all Landsat or all Sentinel',
                QMessageBox.Ok)
            return

        # if sentinel sensors, check resolutions match
        if sentinel:
            res_match = self.check_sentinel2_resolutions_match(
                self.pre_supp_img, self.post_img, self.peri_img)
            if not res_match:
                QMessageBox.critical(
                self, 'Error!',
                'Sentinel resolutions must all be the same, ' +
                'either all 20m, all 30m, or all 60m.',
                QMessageBox.Ok)
                return

        # check bundle levels match for sentinel and landsat
        bundle_match = self.check_bundles_match(
            self.pre_supp_img, self.post_img, self.peri_img)
        if not bundle_match:
            resp = QMessageBox.warning(
                None, 'Warning!',
                ('Bundles with different levels have been selected, '
                'are you sure you wish to use different processing levels?\n'
                'Press OK to continue or '
                'Cancel to choose different inputs.'),
                QMessageBox.Ok, QMessageBox.Cancel)
            if not resp == QMessageBox.Ok:
                return

        # Date Info
        if self.ui.preDatelineEdit.text():
            try:
                prefire_datetime = datetime.datetime.strptime(
                    self.ui.preDatelineEdit.text(), date_format)
            except BaseException:
                QMessageBox.critical(
                    self, 'Error!',
                    'Invalid Prefire Date format. ' +
                    'Expected Format: YYYY-MM-DD',
                    QMessageBox.Ok)
                return
            self.prefire_date = self.ui.preDatelineEdit.text()
        else:
            self.prefire_date = None
        if self.ui.postDatelineEdit.text():
            try:
                postfire_datetime = datetime.datetime.strptime(
                    self.ui.postDatelineEdit.text(), date_format)
            except BaseException:
                QMessageBox.critical(
                    self, 'Error!',
                    'Invalid Postfire Date format. ' +
                    'Expected Format: YYYY-MM-DD',
                    QMessageBox.Ok)
                return
            self.postfire_date = self.ui.postDatelineEdit.text()
        else:
            self.postfire_date = None
        
        if self.prefire_date and self.postfire_date:
            if prefire_datetime >= postfire_datetime:
                QMessageBox.critical(
                    self, 'Error!',
                    'Prefire date is not before Postfire date.',
                    QMessageBox.Ok)
                return

        if self.ui.periDatelineEdit.text():
            try:
                _ = datetime.datetime.strptime(
                    self.ui.periDatelineEdit.text(), date_format)
            except BaseException:
                QMessageBox.critical(
                    self, 'Error!',
                    'Invalid Perimeter Date format. ' +
                    'Expected Format: YYYY-MM-DD',
                    QMessageBox.Ok)
                return
            self.peri_date = self.ui.periDatelineEdit.text()
        else:
            self.peri_date = None
        QDialog.accept(self)

    def check_mapping_sensors_match(self, pre_sensor, post_sensor, peri_sensor):
        ''' Check the sensors are all landsat or sentinel
        (sentinel will be done in future task)'''
        # This will grab indexes 0,1,2 of sensor array and then grab the
        # second item index 1 in each to get the landsat sensors
        landsat_sensors = [i[1] for i in const.SENSOR_ARRAY[0:4]]
        matching = True
        sentinel = False
        sensors = []
        if pre_sensor:
            sensors.append(pre_sensor)
        if post_sensor:
            sensors.append(post_sensor)
        if peri_sensor:
            sensors.append(peri_sensor)

        # Check sensors are all Landsat
        for sensor in sensors:
            if sensor not in landsat_sensors:
                matching = False

        # If matching is False for Landsat then check for Sentinel
        if not matching:
            # Check sensors are all Sentinel
            matching = True
            sentinel_sensors = [i[1] for i in const.SENSOR_ARRAY[4:6]]
            for sensor in sensors:
                if sensor not in sentinel_sensors:
                    matching = False
                else:
                    sentinel = True
        return matching, sentinel

    def check_sentinel2_resolutions_match(self, pre_image, post_image,
                                          peri_image):
        ''' If sentinel2 sensors check the images are all same resolution'''
        res_match = True
        images = []
        if pre_image:
            images.append(pre_image)
        if post_image:
            images.append(post_image)
        if peri_image:
            images.append(peri_image)

        resolution = None
        for image in images:
            image_split = image.split('_', 1)
            meters = image_split[1].split('m')[0]
            if resolution:
                if meters != resolution:
                    res_match = False
            else:
                resolution = meters

        return res_match
    
    def check_bundles_match(self, pre_image, post_image,
                            peri_image):
        ''' Check the images are all same level'''
        bundle_match = True
        images = []
        if pre_image:
            images.append(pre_image)
        if post_image:
            images.append(post_image)
        if peri_image:
            images.append(peri_image)

        level = None
        for image in images:
            image_split = image.split('_')
            lvl = image_split[-1][0:2]
            if level:
                if lvl != level:
                    bundle_match = False
            else:
                level = lvl

        return bundle_match


class ChangePreferences(QDialog):
    def __init__(self, ConfigFile_path, img_src, scene_src):
        QDialog.__init__(self)
        self.ui = Ui_Preferences()
        self.ui.setupUi(self)
        self.img_src = img_src
        self.scene_src = scene_src
        
        self.utils = Utilities()
        self.ConfigFile = ConfigFile_path

        self.config = configparser.ConfigParser()
        self.config.read(self.ConfigFile)
        self.load_file_contents()

        self.ui.saveButton.clicked.connect(self.save_file_contents)
        self.ui.imgSrcBrowseButton.clicked.connect(self.browse_img_src)
        self.ui.sceneDirBrowseButton.clicked.connect(self.browse_base_dir)
        self.ui.deleteVectorStyleButton.clicked.connect(self.delete_layer)
        self.ui.addVectorStyleButton.clicked.connect(self.add_layer)
        self.ui.addWMSButton.clicked.connect(self.add_wms_layer)
        self.ui.deleteWMSButton.clicked.connect(self.delete_wms_layer)

    def load_file_contents(self):
        '''Read config file contents and assign to lineEdits'''
        self.ui.ImgSrclineEdit.setText(
            self.config.get('config', 'img_src').strip())
        self.ui.SceneDirlineEdit.setText(
            self.config.get('config', 'base_dir').strip())
        self.list_vector_styles()
        self.list_wms_layers()

    def list_vector_styles(self):
        '''Read config file vectors_styles section in'''
        files_list = self.config.get(
            'config', 'vectors_styles').strip().splitlines()

        self.ui.VectorlistWidget.clear()

        for files in files_list:
            self.ui.VectorlistWidget.addItem(files)

    def list_wms_layers(self):
        '''Read config file wms_layers section in'''
        files_list = self.config.get(
            'config', 'wms_layers').strip().splitlines()

        self.ui.WMSlistWidget.clear()

        for files in files_list:
            self.ui.WMSlistWidget.addItem(files)

    def save_file_contents(self):
        '''Save lineEdit text to the config file'''

        self.config.set('config', 'img_src',
                        f'{self.ui.ImgSrclineEdit.text().replace('%', f'%%')}\n')
        self.config.set('config', 'base_dir',
                        f'{self.ui.SceneDirlineEdit.text().replace('%', f'%%')}\n')
        self.config.set('config', 'ndvi_url',
                        f'{self.config.get('config', 'ndvi_url').strip().replace('%', f'%%')}\n')

        files_list = ""
        for file in range(self.ui.VectorlistWidget.count()):
            files_list += f"\n{self.ui.VectorlistWidget.item(file).text().replace('%', f'%%')}"

        self.config.set('config', 'vectors_styles', files_list)

        wms_layers_list = ""
        for file in range(self.ui.WMSlistWidget.count()):
            wms_layers_list += f"\n{self.ui.WMSlistWidget.item(file).text().replace('%', f'%%')}"

        self.config.set('config', 'wms_layers', wms_layers_list)

        with open(self.ConfigFile, 'w') as configfile:
            self.config.write(configfile)

    def browse_img_src(self):
        '''Browse for the img_src directory and adjust line edit'''
        file_name = QFileDialog.getExistingDirectory(
            self,
            u'Select img_scr directory',
            self.img_src)
        if file_name != "":
            self.ui.ImgSrclineEdit.setText(
                self.utils.fix_os_sep_in_path(file_name))

    def browse_base_dir(self):
        '''Browse for the base_dir directory and adjust line edit'''
        file_name = QFileDialog.getExistingDirectory(
            self,
            u'Select base_dir directory',
            self.scene_src)
        if file_name != "":
            self.ui.SceneDirlineEdit.setText(
                self.utils.fix_os_sep_in_path(file_name))

    def delete_layer(self):
        '''Delete the selected layer from the list widget'''
        items = self.ui.VectorlistWidget.selectedItems()
        for item in items:
            remove_item = self.ui.VectorlistWidget.takeItem(
                self.ui.VectorlistWidget.row(item))
            del remove_item

    def add_layer(self):
        '''Add selected vector/style layer to the list widget'''
        add_vector_stype_dlg = AddVectorStyleDialog()
        add_vector_stype_dlg.show()
        if add_vector_stype_dlg.exec_():
            vector_layer = add_vector_stype_dlg.vector_layer
            style_file = add_vector_stype_dlg.style_file
            toc_name = add_vector_stype_dlg.toc_name
            add_vector_style = \
                toc_name + ', ' + vector_layer + ', ' + style_file
            self.ui.VectorlistWidget.addItem(add_vector_style)

    def add_wms_layer(self):
        '''Add selected vector/style layer to the list widget'''
        add_wms_layer_dlg = AddWMSLayerDialog()
        add_wms_layer_dlg.show()
        if add_wms_layer_dlg.exec_():
            layer_name = add_wms_layer_dlg.layer_name
            wms_source_uri = add_wms_layer_dlg.wms_source_uri
            add_wms_layer = layer_name + ', ' + wms_source_uri
            self.ui.WMSlistWidget.addItem(add_wms_layer)
    
    def delete_wms_layer(self):
        '''Delete the selected layer from the list widget'''
        items = self.ui.WMSlistWidget.selectedItems()
        for item in items:
            remove_item = self.ui.WMSlistWidget.takeItem(
                self.ui.WMSlistWidget.row(item))
            del remove_item


class AddWMSLayerDialog(QDialog):
    def __init__(self):
        QDialog.__init__(self)
        self.ui = Ui_AddWMSLayer()
        self.ui.setupUi(self)
        self.utils = Utilities()
        self.layer_name = None
        self.wms_source_uri = None
        self.open_wms_file_info =\
            [[layer.name(), layer.source()] for layer
              in QgsProject.instance().mapLayers().values()
              if layer.providerType().lower() == 'wms']

        self.fill_WMS_combo_box()

        self.ui.importButton.clicked.connect(self.import_layer_info)
        self.ui.okButton.clicked.connect(self.check_paths_are_set)

    def import_layer_info(self):
        '''Import selected layer info'''
        for file in self.open_wms_file_info:
            if file[0] == str(self.ui.ImportLayerComboBox.currentText()):
                self.ui.LayerNameLineEdit.setText(file[0])
                self.ui.WMSLineEdit.setText(file[1])

    def fill_WMS_combo_box(self):
        '''Fill the combo box with the open WMS layers that are available
        to import'''
        self.ui.ImportLayerComboBox.clear()
        self.ui.ImportLayerComboBox.addItem("Select from open layers")

        for file in self.open_wms_file_info:
            self.ui.ImportLayerComboBox.addItem(file[0])
            self.ui.ImportLayerComboBox.setCurrentIndex(0)

    def check_or_set_source_uri_for_transparency(self):
        '''Check that the source_uri is a valid type and contains
        "&TRANSPARENT=TRUE" in the string'''
        invalid_types = ['jpeg', 'svg+xml', 'tiff']
        if "&TRANSPARENT=TRUE" not in self.wms_source_uri or\
                "&transparent=TRUE" not in self.wms_source_uri:
            self.wms_source_uri = self.wms_source_uri + "&TRANSPARENT=TRUE"
        
        if '&format=image/' in self.wms_source_uri:
            image_type = self.wms_source_uri.split(
                '&format=image/', 1)[1].split('&',1)[0]
            if image_type.lower() in invalid_types:
                QMessageBox.information(
                    None, r'Image Type Invalid for Transparent Option',
                    'The image type ' + image_type +
                    ' does not allow transparency, changing to png image type',
                    QMessageBox.Ok)
                self.wms_source_uri = self.wms_source_uri.replace(image_type, 'png')
        else:
            QMessageBox.information(
                None, r'Could not find image type in source_uri',
                '"&format=image/" expected in source_uri\n' +
                'As a result we are unable to check that the image type is ' +
                'compatable with the transparent option (png, pn8, gif)\n' +
                'The layer will still be added, check it is correct.',
                QMessageBox.Ok)
        

    def check_paths_are_set(self):
        '''Check both layer name and wms source uri are set'''
        if self.ui.LayerNameLineEdit.text().strip() != '':
            self.layer_name = self.ui.LayerNameLineEdit.text().strip()
        else:
            self.layer_name = None
        if self.ui.WMSLineEdit.text().strip() != '':
            self.wms_source_uri = self.ui.WMSLineEdit.text().strip()
        else:
            self.wms_source_uri = None
        if self.layer_name is None or self.wms_source_uri is None:
            QMessageBox.critical(
                self, 'Error!',
                'Must give a layer_name and wms_source_uri.',
                QMessageBox.Ok)
            return
        if self.ui.TransparentCheckBox.isChecked():
            # Check that the source_uri is a valid type and contains
            # &TRANSPARENT=TRUE
            self.check_or_set_source_uri_for_transparency()
        QDialog.accept(self)


class AddVectorStyleDialog(QDialog):
    def __init__(self):
        QDialog.__init__(self)
        self.ui = Ui_AddVectorStyle()
        self.ui.setupUi(self)
        self.utils = Utilities()
        self.style_dir = os.path.join(
            os.path.dirname(os.path.realpath(__file__)), 'Style')
        self.vector_dir = os.path.join(
            os.path.dirname(os.path.realpath(__file__)), 'Vector_Files')
        self.vector_layer = None
        self.style_file = None
        self.toc_name = None

        self.ui.vectorBrowseButton.clicked.connect(self.browse_vector_file)
        self.ui.styleBrowseButton.clicked.connect(self.browse_style_file)
        self.ui.okButton.clicked.connect(self.check_paths_are_set)

    def browse_vector_file(self):
        '''Browse to the vector layer and adjust line edit'''
        file_name = QFileDialog.getOpenFileName(
            self,
            u'Select vector layer',
            self.vector_dir,
            "Shapefiles (*.shp)")[0]
        if file_name != "":
            self.ui.VectorlineEdit.setText(
                self.utils.fix_os_sep_in_path(file_name))
            self.vector_layer = self.ui.VectorlineEdit.text()
            self.ui.NamelineEdit.setText(
                os.path.basename(self.ui.VectorlineEdit.text()).split(
                    '.shp', 1)[0])
            self.toc_name = self.ui.NamelineEdit.text()

    def browse_style_file(self):
        '''Browse to the vector layer and adjust line edit'''
        file_name = QFileDialog.getOpenFileName(
            self,
            u'Select style file',
            self.style_dir,
            "Styles (*.qml *.clr)")[0]
        if file_name != "":
            self.ui.StylelineEdit.setText(
                self.utils.fix_os_sep_in_path(file_name))
            self.style_file = self.ui.StylelineEdit.text()

    def check_paths_are_set(self):
        '''Check vector and style files and TOC name is set'''
        if self.ui.NamelineEdit.text().strip() != '':
            self.toc_name = self.ui.NamelineEdit.text()
        else:
            self.toc_name = None
        if self.style_file is None or self.vector_layer is None \
            or self.toc_name is None:
            QMessageBox.critical(
                self, 'Error!',
                'Must select both a vector and style file.',
                QMessageBox.Ok)
            return
        QDialog.accept(self)
