import tkinter as tk
from tkinter import filedialog, messagebox, ttk
from ssebop_espa.processing_steps import process_espa_files, gapfill_ETf, interpolate_espa_ETf, monthly_ETa, \
    seasonal_ETa, annual_ETa


class App:
    def __init__(self, root):
        self.root = root
        self.root.title('SSEBop Menu')

        self.tab_control = ttk.Notebook(root)

        self.output_dir_var_process = tk.StringVar()  # Output directory for process ESPA files
        self.input_dir_var_gapfill = tk.StringVar()  # Input directory for gapfill, pre-filled with process ESPA files output
        self.default_interp_path = tk.StringVar()
        self.default_interp_days = 48
        self.create_tabs()

        self.tab_control.pack(expand=1, fill='both')

    def create_tabs(self):
        # Process ESPA Files Tab
        tab_process = ttk.Frame(self.tab_control)
        self.tab_control.add(tab_process, text="Process ESPA Files (1)")
        self.setup_tab_process(tab_process)

        # Gapfill ETf Tab
        tab_gapfill = ttk.Frame(self.tab_control)
        self.tab_control.add(tab_gapfill, text="Gapfill ETf (2)")
        self.setup_tab_gapfill(tab_gapfill)

        # Interpolate ESPA ETf Tab
        tab_interpolate = ttk.Frame(self.tab_control)
        self.tab_control.add(tab_interpolate, text="Interpolate ESPA ETf (3)")
        self.setup_tab_interpolate(tab_interpolate)

        # Aggregate ET Tab
        tab_aggregate = ttk.Frame(self.tab_control)
        self.tab_control.add(tab_aggregate, text="Aggregate ET (4)")
        self.setup_tab_aggregate(tab_aggregate)

    def browse_directory(self, var):
        filename = filedialog.askdirectory()
        var.set(filename)

    def setup_tab_process(self, tab):
        input_dir_var = tk.StringVar()
        status_text = tk.StringVar()
        progress_var = tk.IntVar()

        tk.Label(tab, text="Select Input Directory (Location of .tar.gz files):").grid(row=0, column=0, padx=10, pady=10)
        tk.Entry(tab, textvariable=input_dir_var, width=50).grid(row=0, column=1, columnspan=3, padx=10, pady=10)
        tk.Button(tab, text="Browse", command=lambda: self.browse_directory(input_dir_var)).grid(row=0, column=4,
                                                                                                 padx=10, pady=10)

        progress_bar = ttk.Progressbar(tab, orient="horizontal", length=400, mode="determinate", variable=progress_var)
        progress_bar.grid(row=2, column=0, columnspan=5, padx=10, pady=10)

        tk.Label(tab, text="Status:").grid(row=3, column=0, padx=10, pady=10)
        tk.Label(tab, textvariable=status_text, width=50).grid(row=3, column=1, columnspan=3, padx=10, pady=10)

        tk.Button(tab, text="Run Script",
                  command=lambda: self.run_script_process(input_dir_var.get(), progress_var, status_text)).grid(row=4,
                                                                                                                column=0,
                                                                                                                columnspan=5,
                                                                                                                padx=10,
                                                                                                                pady=20)

    def setup_tab_gapfill(self, tab):
        status_text = tk.StringVar()
        progress_var = tk.IntVar()
        interp_days_var = tk.IntVar()

        tk.Label(tab, text="Select Input Directory (..data/pathxxrowyy):").grid(row=0, column=0, padx=10, pady=10)
        tk.Entry(tab, textvariable=self.input_dir_var_gapfill, width=50).grid(row=0, column=1, columnspan=3, padx=10,
                                                                              pady=10)
        tk.Button(tab, text="Browse", command=lambda: self.browse_directory(self.input_dir_var_gapfill)).grid(row=0,
                                                                                                              column=4,
                                                                                                              padx=10,
                                                                                                              pady=10)

        tk.Label(tab, text="Interpolation Days To Search Before and After Raster:").grid(row=1, column=0, padx=10, pady=10)
        tk.Entry(tab, textvariable=interp_days_var, width=10).grid(row=1, column=1, padx=10, pady=10)

        progress_bar = ttk.Progressbar(tab, orient="horizontal", length=400, mode="determinate", variable=progress_var)
        progress_bar.grid(row=2, column=0, columnspan=5, padx=10, pady=10)

        tk.Label(tab, text="Status:").grid(row=3, column=0, padx=10, pady=10)
        tk.Label(tab, textvariable=status_text, width=50).grid(row=3, column=1, columnspan=3, padx=10, pady=10)

        tk.Button(tab, text="Run Script",
                  command=lambda: self.run_script_gapfill(self.input_dir_var_gapfill.get(), progress_var,
                                                          status_text, interp_days_var.get())).grid(row=4, column=0, columnspan=5, padx=10,
                                                                             pady=20)

    def setup_tab_interpolate(self, tab):
        pet_file_fmt = tk.StringVar()
        pet_dir = tk.StringVar()
        input = tk.StringVar()
        status_text = tk.StringVar()
        progress_var = tk.IntVar()
        bool_var = tk.StringVar(value="True")

        # INPUT DIRECTORY
        tk.Label(tab, text="Select Input Directory (..data/pathxxrowyy):").grid(row=0, column=0, padx=10, pady=10)
        tk.Entry(tab, textvariable=input, width=50).grid(row=0, column=1, columnspan=3, padx=10, pady=10)
        tk.Button(tab, text="Browse", command=lambda: self.browse_directory(input)).grid(row=0, column=4,
                                                                                                 padx=10, pady=10)

        # PET DIRECTORY
        tk.Label(tab, text="Select Climatology Reference ET Directory:").grid(row=1, column=0, padx=10, pady=10)
        tk.Entry(tab, textvariable=pet_dir, width=50).grid(row=1, column=1, columnspan=3, padx=10,
                                                                              pady=10)
        tk.Button(tab, text="Browse", command=lambda: self.browse_directory(pet_dir)).grid(row=1, column=4,
                                                                                           padx=10, pady=10)
        # PET FILE FORMAT
        tk.Label(tab, text="Reference ET file fmt ('pet_ddd'):").grid(row=2, column=0, padx=10, pady=10)
        tk.Entry(tab, textvariable=pet_file_fmt, width=50).grid(row=2, column=1, columnspan=3, padx=10,
                                                           pady=10)

        # TRUE/FALSE for Climatology
        tk.Label(tab, text="Is daily Reference ET a Climatology ('true'):").grid(row=3, column=0, padx=10, pady=10)
        option_menu = ttk.OptionMenu(tab, bool_var, "True", "True", "False")
        option_menu.grid(row=3, column=1, padx=10, pady=10)


        progress_bar = ttk.Progressbar(tab, orient="horizontal", length=400, mode="determinate", variable=progress_var)
        progress_bar.grid(row=4, column=0, columnspan=5, padx=10, pady=10)

        tk.Label(tab, text="Status:").grid(row=5, column=0, padx=10, pady=10)
        tk.Label(tab, textvariable=status_text, width=50).grid(row=5, column=1, columnspan=3, padx=10, pady=10)

        tk.Button(tab, text="Run Script",
                  command=lambda: self.run_script_interpolate(input.get(),
                                                              pet_dir.get(), pet_file_fmt.get(), bool_var.get(),
                                                              progress_var, status_text)).grid(row=6, column=0,
                                                                                               columnspan=5, padx=10,
                                                                                               pady=20)

    def setup_tab_aggregate(self, tab):
        # Create a Notebook widget for sub-tabs
        sub_tab_control = ttk.Notebook(tab)

        # Monthly ETa Sub-tab
        sub_tab_monthly = ttk.Frame(sub_tab_control)
        sub_tab_control.add(sub_tab_monthly, text="Monthly ETa")
        self.setup_sub_tab_monthly(sub_tab_monthly)

        # Annual ETa Sub-tab
        sub_tab_annual = ttk.Frame(sub_tab_control)
        sub_tab_control.add(sub_tab_annual, text="Annual ETa")
        self.setup_sub_tab_annual(sub_tab_annual)

        # Seasonal ETa Sub-tab
        sub_tab_seasonal = ttk.Frame(sub_tab_control)
        sub_tab_control.add(sub_tab_seasonal, text="Seasonal ETa")
        self.setup_sub_tab_seasonal(sub_tab_seasonal)

        sub_tab_control.pack(expand=1, fill='both')

    def setup_sub_tab_monthly(self, tab):
        """Makes monthly data for a given year selected by the user"""

        input_dir_var = tk.StringVar()
        year_var = tk.IntVar()
        status_text = tk.StringVar()
        progress_var = tk.IntVar()

        tk.Label(tab, text="Select Input Directory ('..data/pathXXrowYY/dailyETa'):").grid(row=0, column=0, padx=10, pady=10)
        tk.Entry(tab, textvariable=input_dir_var, width=50).grid(row=0, column=1, columnspan=3, padx=10, pady=10)
        tk.Button(tab, text="Browse", command=lambda: self.browse_directory(input_dir_var)).grid(row=0, column=4,
                                                                                                 padx=10, pady=10)

        tk.Label(tab, text="Year:").grid(row=1, column=0, padx=10, pady=10)
        tk.Entry(tab, textvariable=year_var, width=10).grid(row=1, column=1, padx=10, pady=10)

        progress_bar = ttk.Progressbar(tab, orient="horizontal", length=400, mode="determinate", variable=progress_var)
        progress_bar.grid(row=2, column=0, columnspan=5, padx=10, pady=10)

        tk.Label(tab, text="Status:").grid(row=3, column=0, padx=10, pady=10)
        tk.Label(tab, textvariable=status_text, width=50).grid(row=3, column=1, columnspan=3, padx=10, pady=10)

        tk.Button(tab, text="Run Script",
                  command=lambda: self.run_script_monthly(input_dir_var.get(), year_var.get(),
                                                          progress_var, status_text)).grid(row=4, column=0,
                                                                                           columnspan=5, padx=10,
                                                                                           pady=20)

    def setup_sub_tab_annual(self, tab):
        input_dir_var = tk.StringVar()
        year = tk.IntVar()
        status_text = tk.StringVar()
        progress_var = tk.IntVar()

        tk.Label(tab, text="Select Input Directory ('..data/pathXXrowYY/dailyETa'):").grid(row=0, column=0, padx=10, pady=10)
        tk.Entry(tab, textvariable=input_dir_var, width=50).grid(row=0, column=1, columnspan=3, padx=10, pady=10)
        tk.Button(tab, text="Browse", command=lambda: self.browse_directory(input_dir_var)).grid(row=0, column=4,
                                                                                                 padx=10, pady=10)

        tk.Label(tab, text="Select Year:").grid(row=1, column=0, padx=10, pady=10)
        tk.Entry(tab, textvariable=year, width=50).grid(row=1, column=1, columnspan=3, padx=10, pady=10)

        progress_bar = ttk.Progressbar(tab, orient="horizontal", length=400, mode="determinate", variable=progress_var)
        progress_bar.grid(row=2, column=0, columnspan=5, padx=10, pady=10)

        tk.Label(tab, text="Status:").grid(row=3, column=0, padx=10, pady=10)
        tk.Label(tab, textvariable=status_text, width=50).grid(row=3, column=1, columnspan=3, padx=10, pady=10)

        tk.Button(tab, text="Run Script",
                  command=lambda: self.run_script_annual(input_dir_var.get(), year.get(), progress_var,
                                                         status_text)).grid(row=4, column=0, columnspan=5, padx=10,
                                                                            pady=20)

    def setup_sub_tab_seasonal(self, tab):
        input_dir_var = tk.StringVar()
        year_var = tk.StringVar()
        start_month_var = tk.StringVar()
        end_month_var = tk.StringVar()
        status_text = tk.StringVar()
        progress_var = tk.IntVar()

        tk.Label(tab, text="Select Input Directory ('..data/pathXXrowYY/dailyETa'):").grid(row=0, column=0, padx=10, pady=10)
        tk.Entry(tab, textvariable=input_dir_var, width=50).grid(row=0, column=1, columnspan=3, padx=10, pady=10)
        tk.Button(tab, text="Browse", command=lambda: self.browse_directory(input_dir_var)).grid(row=0, column=4,
                                                                                                 padx=10, pady=10)

        tk.Label(tab, text="Start Year:").grid(row=1, column=0, padx=10, pady=10)
        tk.Entry(tab, textvariable=year_var, width=10).grid(row=1, column=1, padx=10, pady=10)

        tk.Label(tab, text="Start Month:").grid(row=2, column=0, padx=10, pady=10)
        tk.Entry(tab, textvariable=start_month_var, width=10).grid(row=2, column=1, padx=10, pady=10)

        tk.Label(tab, text="End Month:").grid(row=3, column=0, padx=10, pady=10)
        tk.Entry(tab, textvariable=end_month_var, width=10).grid(row=3, column=1, padx=10, pady=10)

        progress_bar = ttk.Progressbar(tab, orient="horizontal", length=400, mode="determinate", variable=progress_var)
        progress_bar.grid(row=4, column=0, columnspan=5, padx=10, pady=10)

        tk.Label(tab, text="Status:").grid(row=5, column=0, padx=10, pady=10)
        tk.Label(tab, textvariable=status_text, width=50).grid(row=5, column=1, columnspan=3, padx=10, pady=10)

        tk.Button(tab, text="Run Script",
                  command=lambda: self.run_script_seasonal(input_dir_var.get(), start_month_var.get(),
                                                           end_month_var.get(), year_var.get(),
                                                           progress_var, status_text)).grid(row=6, column=0,
                                                                                            columnspan=5, padx=10,
                                                                                            pady=20)

    def run_script_process(self, input_dir, progress_var, status_text):
        if not input_dir:
            messagebox.showerror("Error", "Please select both input and output directories")
            return

        def update_progress(value):
            progress_var.set(value)
            self.root.update_idletasks()

        def update_status(message):
            status_text.set(message)
            self.root.update_idletasks()

        progress_var.set(0)
        output_dir = process_espa_files(input_dir, update_progress, update_status, self.root)
        self.output_dir_var_process.set(output_dir)
        self.input_dir_var_gapfill.set(output_dir)
        messagebox.showinfo("Info", "Process ESPA Files executed successfully")

    def run_script_gapfill(self, input_dir, progress_var, status_text, gapfill_days):
        if not input_dir:
            messagebox.showerror("Error", "Please select input directory")
            return

        def update_progress(value):
            progress_var.set(value)
            self.root.update_idletasks()

        def update_status(message):
            status_text.set(message)
            self.root.update_idletasks()

        progress_var.set(0)
        gapfill_ETf(input_dir, update_progress, update_status, self.root, limit=gapfill_days)
        messagebox.showinfo("Info", "Gapfill ETf executed successfully")

    def run_script_interpolate(self, input_dir, pet_dir, pet_name_fmt, climatology,  progress_var, status_text):

        print(f'input dir {input_dir}, pet dir {pet_dir}, pet name fmt {pet_name_fmt}'
        )

        if not input_dir:
            messagebox.showerror("Error", "Please complete all fields")
            return

        def update_progress(value):
            progress_var.set(value)
            self.root.update_idletasks()

        def update_status(message):
            status_text.set(message)
            self.root.update_idletasks()

        progress_var.set(0)
        interpolate_espa_ETf(input_dir, pet_dir, pet_name_fmt, climatology, update_progress,
                             update_status, self.root)
        messagebox.showinfo("Info", "Interpolate ESPA ETf executed successfully")

    def run_script_monthly(self, input_dir, year, progress_var, status_text):

        if not input_dir:
            messagebox.showerror("Error", "Please complete all fields")
            return

        def update_progress(value):
            progress_var.set(value)
            self.root.update_idletasks()

        def update_status(message):
            status_text.set(message)
            self.root.update_idletasks()

        progress_var.set(0)
        monthly_ETa(input_dir, year, update_progress, update_status, root=self.root)
        messagebox.showinfo("Info", "Monthly ETa executed successfully")

    def run_script_annual(self, input_dir, year, progress_var, status_text):
        if not input_dir:
            messagebox.showerror("Error", "Please complete all fields")
            return

        def update_progress(value):
            progress_var.set(value)
            self.root.update_idletasks()

        def update_status(message):
            status_text.set(message)
            self.root.update_idletasks()

        progress_var.set(0)
        annual_ETa(input_dir, year, progress=update_progress, status_msg=update_status, root=self.root)
        messagebox.showinfo("Info", "Annual ETa executed successfully")

    def run_script_seasonal(self, input_dir, start_month, end_month, year, progress_var, status_text):
        if not input_dir:
            messagebox.showerror("Error", "Please complete all fields")
            return

        def update_progress(value):
            progress_var.set(value)
            self.root.update_idletasks()

        def update_status(message):
            status_text.set(message)
            self.root.update_idletasks()

        progress_var.set(0)
        seasonal_ETa(input_dir, int(start_month), int(end_month), year, progress=update_progress, status_msg=update_status, root=self.root)
        messagebox.showinfo("Info", "Seasonal ETa executed successfully")

def main():
    root = tk.Tk()
    app = App(root)
    root.mainloop()

if __name__ == '__main__':
    main()
